from typing import Union, Optional, List, Dict, Any
def build_planner_prompt() -> str:
    return """
You are a Planner in an image quality assessment (IQA) Agent system. Your task is to analyze the user's query and generate a structured plan for downstream assessment.

Return a valid JSON object in the following format:
{
  "task_type": "IQA" or "Others",
  "reference_type": "Full-Reference" or "No-Reference",
  "required_object_names": ["<object1>", "<object2>", ...] or null,
  "required_distortions": {
    "<object_name or Global>": ["<distortion_1>", "<distortion_2>", ...]
  } or null,
  "required_tools": ["<tool_name_1>", "<tool_name_2>", ...] or null,
  "distortion_source": "explicit" or "inferred",
  "plan": {
    "distortion_detection": true or false,
    "tool_selection": true or false,
    "distortion_analysis": true or false,
    "tool_execute": true or false
  }
}

Instructions:

1. Task Type:
- If the query is related to technical distortions, set "task_type" to "IQA".
- Else, if the query is related to aesthetic-related other low-level attributes (e.g., composition, style, emotion), set "task_type" to "Others".

2. Reference Type:
- If both distorted and reference images are present, set "reference_type" to "Full-Reference".
- Otherwise, set to "No-Reference".

3. Required Object Names:
- Extract object or region names from the query (e.g., "the building", "purple flowers", "the sky").
- Set "required_object_names" to a list of names, or null if none are mentioned.

4. Distortion Source:
- If the query clearly mentions distortion types, set "distortion_source" to "explicit".
- Otherwise, set it to "inferred".

5. Required Distortions:
- If distortions apply to specific objects, use object names as keys, and fill the "required_distortions" field accordingly.
- If distortions apply to the whole image, use "Global" as key, and fill the "required_distortions" field accordingly
- If no distortion is mentioned, set "required_distortions" to null.

6. Required Tools:
- Only include tool names if explicitly mentioned by the user (e.g., "use LPIPS").
- Otherwise, set "required_tools" to null.

7. Plan:
- If "task_type" is "Others", set "distortion_detection" to false, "distortion_analysis" to false, set "tool_selection" to false, set "tool_execute" to false.
- If "distortion_source" is "explicit", set "distortion_detection" to false, "distortion_analysis" to true.
- If "distortion_source" is "inferred", set "distortion_detection" to true, "distortion_analysis" to true.
- If the query refers to specific regions or objects, set "tool_selection" to false and set "tool_execute" to false.
- If the query is about the whole image (global-level), set "tool_selection" to true and "tool_execute" to true.
- If "Required Tools" is not null/none, set "tool_selection" to false and "tool_execute" to true.

Only return valid JSON. Do not include explanations, markdown, or extra text.
"""

def build_distortion_detection_prompt(object_level: bool,user_question: str) -> str:
    # distortion_types = ['Brightness', 'Colorfulness', 'Contrast', 'Noisiness', 'Sharpness']
    distortion_types = ['Blurs', 'Color distortions', 'Compression', 'Noise', 'Brightness change', 'Spatial distortions', 'Sharpness and contrast']
    region_instr = (
        "Focus only on the specified regions. Compare each region with the reference image if provided."
        if object_level else
        "Analyze the entire distorted image. Compare it with the reference image if provided."
    )

    prompt = f"""
        User question: "{user_question}"
        
        You are a distortion analysis expert. 
        
        Based on the following **user question**, identify all possible distortions need to be focused on to properly address the user's intent.

        Select the distortions from the following list: {distortion_types}

        Output format (valid JSON only):

        {{
            "<object_name or 'Global'>": ["<distortion_type_1>", "<distortion_type_2>"]
        }}

        Instructions:
        1. {region_instr}
        2. Analyze each of the distortion types listed above.
        3. Only return valid JSON. Do not include extra text or markdown.
        """
    return prompt

def build_distortion_analysis_prompt_multi_object(
    distortion_dict: Dict[str, List[str]],
    has_reference: bool = False,
    user_question: Optional[str] = ""
) -> str:
    region_blocks = []
    for obj_name, dist_list in distortion_dict.items():
        dist_str = ", ".join(dist_list)
        region_blocks.append(f"- Region: {obj_name}\n  Distortions to analyze: {dist_str}")
    full_region_description = "\n\n".join(region_blocks)

    image_line = (
        "You are given a distorted image and a reference image. "
        "Compare the two to assess the distortion impact."
        if has_reference else
        "You are given only a distorted image. Analyze the distortions based on visible visual cues."
    )

    prompt = f"""
You are a distortion analysis expert. Your task is to assess the **severity and visual impact** of various distortion types for different regions of an image.

User question: "{user_question}"

{image_line}

The image contains multiple distinct regions, and each region is associated with its own set of distortion types.

Please analyze the following regions independently:

{full_region_description}

Return your output as valid JSON in the format:

{{
    "<object_name or 'Global'>": [
      {{
        "type": "<distortion_name>",
        "severity": "<none/mild/moderate/heavy/severe>",
        "explanation": "<brief visual impact description>"
      }}
    ],
    ...
  }}


Instructions:
1. Base your analysis on the listed distortion types and consider the user question.
2. Use "none" if a distortion is barely or not visible.
3. Use the following severity levels NOT using numbers: 
        - severity = 1 → "none"
        - severity = 2 → "mild"
        - severity = 3 → "moderate"
        - severity = 4 → "heavy"
        - severity = 5 → "severe"
4. In the `explanation` field, briefly describe **where** the distortion appears (e.g., background, face, edges) and **how** it affects the perceived image quality (e.g., reduces sharpness, causes color imbalance, introduces blocky artifacts). Keep the explanation concise and focused on visible visual impact. **Do not identify specific people.**
5. Do not mix results between regions.
6. Only return valid JSON. No explanations, markdown, or comments.
""".strip()
    return prompt

def build_required_tool_prompt(tool_name:str) -> str:
    prompt = f"""
    You are a tool executor. Your task is to execute the {tool_name} required by user and return the score.

    Return your output in the following JSON format:
    {{
        "computed_score": <float between 1 and 5>,
        }}

    Only return valid JSON. Do not include extra text and markdown.
    """
    return prompt

def build_summary_choice_prompt(choices: List[str], user_question: str) -> str:
    choice_letters = [chr(65 + i) for i in range(len(choices))]
    letter_str = ", ".join(choice_letters)

    return f"""
You are a visual quality assessment assistant.

User question: "{user_question}"

Your task is to select the most appropriate answer to the user's question. You are given:
- Distortion analysis (severity and visual impact of listed distortions)
- Tool response (overall quality scores from IQA models)
- Image content

Decision process:
1. First, understand what kind of visual information is needed to answer the user's question.
2. Check if the provided distortion analysis or tool response already contains the required information.
3. If the provided information is sufficient, use it to answer.
4. If the information is unclear or insufficient, analyze the image directly to determine the best answer.

Answer must be one of the following choices: {letter_str}

Return your answer strictly in this JSON format:
{{
  "final_answer": "<one of the above letters>",
  "quality_reasoning": "<brief explanation, based on either distortion analysis, tool response, or direct visual observation>"
}}

""".strip()

def build_summary_others_prompt(choices: List[str]) -> str:
    choice_letters = [chr(65 + i) for i in range(len(choices))]
    letter_str = ", ".join(choice_letters)
    return f"""
    You are an assistant for image quality assessment.

    Your task is to answer the user's question by directly analyzing the given image.

    Carefully examine the image and choose the option that best answers the question, based on what you see.

    The answer must be one of the following choices: {letter_str}.

    Return your answer strictly in this JSON format:
    {{
        "final_answer": "<one of the above letters>",
        "quality_reasoning": "<brief explanation of why this choice was selected>"
    }}

    Only return the JSON object. Do not include extra text or formatting.
    """

def build_summary_nochoice_prompt() -> str:
    prompt = """
    You are an image quality assessor.

    Given the question and the analysis (tool scores, distortion analysis), assess the image quality.

    You must select **one single answer** from the following:
    A. Excellent
    B. Good
    C. Fair
    D. Poor
    E. Bad
    (from high to low).
    Return your answer using ONLY the letter A, B, C, D, or E — no extra explanation or formatting.
    """
    return prompt

from react_tool_config import TOOL_DESCRIPTIONS

full_reference_tools = [
    "TopIQ_FR_tool", "AHIQ_tool", "FSIM_tool", "LPIPS_tool", "DISTS_tool", "WaDIQaM_FR_tool",
    "PieAPP_tool", "MS_SSIM_tool", "GMSD_tool", "SSIM_tool", "CKDN_tool", "VIF_tool", "PSNR_tool", "VSI_tool"
]

no_reference_tools = [
    "QAlign_tool", "CLIPIQA_tool", "UNIQUE_tool", "HyperIQA_tool", "TReS_tool", "WaDIQaM_NR_tool",
    "DBCNN_tool", "ARNIQA_tool", "NIMA_tool", "BRISQUE_tool", "NIQE_tool", "MANIQA_tool", "LIQE_mix_tool"
]

def build_tool_prompt(reference_type: str) -> str:
    if reference_type == "Full-Reference":
        tool_names = full_reference_tools
    else:
        tool_names = no_reference_tools
    descriptions = "\n".join([
        f"- {TOOL_DESCRIPTIONS[name]['name']}: {TOOL_DESCRIPTIONS[name]['description']}"
        for name in tool_names if name in TOOL_DESCRIPTIONS
    ])

    prompt = f"""
    You are a tool executor.
    
    Your task is to assign the most appropriate IQA tool to each visual distortion type, based on the descriptions of the available tools.

    Available tools:
    {descriptions}

    Return your output in the following JSON format:
    {{
        "<object_name or 'Global'>": {{
        "<distortion_name_1>": "<tool_name>",
        "<distortion_name_2>": "<tool_name>"
        }}
    }}

    Instructions:
    1. For each distortion, choose the tool whose description suggests it performs best for that type of distortion. 
    2. If no exact match exists, select the most semantically relevant one.
    3. Tool names must exactly match the provided names. Do NOT add extra prefixes like "functions.".
    4. Return only valid JSON. Do not include explanations, markdown, or extra text.
    """
    return prompt

